<?php
/**
 * Pashizi Widget Integration Examples
 * 
 * This file demonstrates various ways to integrate the Pashizi Currency Widget
 * into your WordPress theme or plugin.
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Example 1: Basic shortcode usage in content
 */
function example_basic_widget() {
    return do_shortcode('[pashizi_widget]');
}

/**
 * Example 2: Widget with custom parameters
 */
function example_custom_widget() {
    return do_shortcode('[pashizi_widget currency="eur" lang="fa" class="my-widget"]');
}

/**
 * Example 3: Multiple widgets for different currencies
 */
function example_multiple_widgets() {
    $output = '<div class="currency-widgets-grid">';
    
    $currencies = ['usd', 'eur', 'gbp', 'cad'];
    
    foreach ($currencies as $currency) {
        $output .= '<div class="widget-column">';
        $output .= do_shortcode('[pashizi_widget currency="' . $currency . '" lang="en"]');
        $output .= '</div>';
    }
    
    $output .= '</div>';
    
    return $output;
}

/**
 * Example 4: Conditional widget display based on user location or preference
 */
function example_conditional_widget() {
    // Get user's preferred language (example using WordPress locale)
    $user_locale = get_locale();
    $lang = 'en'; // default
    
    // Map WordPress locale to widget language
    $locale_map = [
        'fa_IR' => 'fa',
        'ar' => 'ar',
        'de_DE' => 'de',
        'es_ES' => 'es',
        'fr_FR' => 'fr',
        'it_IT' => 'it',
        'nl_NL' => 'nl',
        'ru_RU' => 'ru',
        'zh_CN' => 'zh'
    ];
    
    if (isset($locale_map[$user_locale])) {
        $lang = $locale_map[$user_locale];
    }
    
    return do_shortcode('[pashizi_widget currency="usd" lang="' . $lang . '"]');
}

/**
 * Example 5: Widget with dynamic currency based on post meta or custom field
 */
function example_dynamic_currency_widget($post_id = null) {
    if (!$post_id) {
        $post_id = get_the_ID();
    }
    
    // Get currency from post meta (example: set in a custom field)
    $currency = get_post_meta($post_id, 'featured_currency', true);
    
    // Default to USD if no currency set
    if (!$currency) {
        $currency = 'usd';
    }
    
    return do_shortcode('[pashizi_widget currency="' . $currency . '" lang="en"]');
}

/**
 * Example 6: Add widget to theme programmatically
 */
function example_add_widget_to_theme() {
    // Hook this function to an appropriate action like 'wp_footer' or theme-specific hooks
    echo '<div class="theme-currency-widget">';
    echo do_shortcode('[pashizi_widget currency="usd" lang="fa"]');
    echo '</div>';
}
// add_action('wp_footer', 'example_add_widget_to_theme');

/**
 * Example 7: Widget in WordPress hook/action
 */
function example_widget_in_hook() {
    if (is_single() && in_category('finance')) {
        echo '<div class="post-currency-widget">';
        echo do_shortcode('[pashizi_widget currency="eur" lang="en"]');
        echo '</div>';
    }
}
// add_action('the_content', 'example_widget_in_hook');

/**
 * Example 8: Custom widget wrapper with additional styling
 */
function example_styled_widget($currency = 'usd', $lang = 'en', $title = '') {
    $output = '<div class="pashizi-widget-wrapper">';
    
    if ($title) {
        $output .= '<h3 class="widget-title">' . esc_html($title) . '</h3>';
    }
    
    $output .= '<div class="widget-content">';
    $output .= do_shortcode('[pashizi_widget currency="' . $currency . '" lang="' . $lang . '"]');
    $output .= '</div>';
    
    $output .= '<div class="widget-footer">';
    $output .= '<p><small>Powered by <a href="https://www.pashizi.com" target="_blank">Pashizi</a></small></p>';
    $output .= '</div>';
    
    $output .= '</div>';
    
    return $output;
}

/**
 * Example 9: AJAX-loaded widget (for performance optimization)
 */
function example_ajax_widget() {
    // Return placeholder that will be populated via AJAX
    return '<div id="pashizi-ajax-widget" data-currency="usd" data-lang="en">Loading currency data...</div>';
}

function example_ajax_load_widget() {
    $nonce = isset($_POST['_wpnonce']) ? sanitize_text_field(wp_unslash($_POST['_wpnonce'])) : '';
    if (empty($nonce) || !wp_verify_nonce($nonce, 'load_pashizi_widget')) {
        wp_die(esc_html__('Nonce verification failed', 'pashizi-widget'));
    }
    $currency = isset($_POST['currency']) ? sanitize_text_field(wp_unslash($_POST['currency'])) : 'usd';
    $lang = isset($_POST['lang']) ? sanitize_text_field(wp_unslash($_POST['lang'])) : 'en';
    echo do_shortcode('[pashizi_widget currency="' . $currency . '" lang="' . $lang . '"]');
    wp_die();
}
// add_action('wp_ajax_load_pashizi_widget', 'example_ajax_load_widget');
// add_action('wp_ajax_nopriv_load_pashizi_widget', 'example_ajax_load_widget');

/**
 * Example 10: Widget with custom CSS classes based on currency
 */
function example_currency_themed_widget($currency = 'usd') {
    $currency_classes = [
        'usd' => 'currency-usd american-style',
        'eur' => 'currency-eur european-style',
        'gbp' => 'currency-gbp british-style',
        'jpy' => 'currency-jpy japanese-style'
    ];
    
    $class = isset($currency_classes[$currency]) ? $currency_classes[$currency] : 'currency-default';
    
    return do_shortcode('[pashizi_widget currency="' . $currency . '" lang="en" class="' . $class . '"]');
}

/**
 * Example 11: Responsive widget grid
 */
function example_responsive_widget_grid() {
    $currencies = ['usd', 'eur', 'gbp', 'cad', 'aud', 'chf'];
    
    $output = '<div class="pashizi-responsive-grid">';
    
    foreach ($currencies as $currency) {
        $output .= '<div class="grid-item">';
        $output .= do_shortcode('[pashizi_widget currency="' . $currency . '" lang="en"]');
        $output .= '</div>';
    }
    
    $output .= '</div>';
    
    // Add responsive CSS
    $output .= '<style>
    .pashizi-responsive-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
        gap: 20px;
        margin: 20px 0;
    }
    .grid-item {
        background: #f9f9f9;
        padding: 15px;
        border-radius: 8px;
        border: 1px solid #ddd;
    }
    </style>';
    
    return $output;
}

/**
 * Example 12: Widget shortcode with all parameters
 */
function example_full_featured_widget() {
    return do_shortcode('[pashizi_widget currency="eur" lang="fa" class="premium-widget featured-currency" id="homepage-currency-widget"]');
}

// CSS example for custom styling
/*
.premium-widget {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    border-radius: 15px;
    padding: 20px;
    box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
}

.featured-currency {
    border: 3px solid #gold;
    position: relative;
}

.featured-currency::before {
    content: "Featured";
    position: absolute;
    top: -10px;
    left: 20px;
    background: #gold;
    color: black;
    padding: 5px 10px;
    font-size: 12px;
    border-radius: 15px;
}
*/

/**
 * Example usage in theme files:
 * 
 * In header.php:
 * <?php echo example_conditional_widget(); ?>
 * 
 * In sidebar.php:
 * <?php echo example_styled_widget('usd', 'en', 'Live Exchange Rates'); ?>
 * 
 * In single.php (for finance posts):
 * <?php echo example_dynamic_currency_widget(); ?>
 * 
 * In footer.php:
 * <?php echo example_responsive_widget_grid(); ?>
 */
