<?php
/**
 * Pashizi Currency Widget Class
 */
class Pashizi_Currency_Widget extends WP_Widget {
    
    /**
     * Constructor
     */
    public function __construct() {
        parent::__construct(
            'pashizi_widget',
            esc_html__('Pashizi Currency Widget', 'pashizi-widget'),
            array(
                'description' => esc_html__('Display real-time currency exchange rates with an interactive calculator.', 'pashizi-widget'),
                'classname' => 'pashizi-currency-widget',
            )
        );
    }
    
    /**
     * Widget output
     */
    public function widget($args, $instance) {
    echo wp_kses_post($args['before_widget']);
        
        if (!empty($instance['title'])) {
            echo wp_kses_post($args['before_title']) . esc_html(apply_filters('widget_title', $instance['title'])) . wp_kses_post($args['after_title']);
        }
        
        $currency = !empty($instance['currency']) ? $instance['currency'] : 'usd';
        $lang = !empty($instance['lang']) ? $instance['lang'] : 'en';
        $custom_class = !empty($instance['custom_class']) ? $instance['custom_class'] : '';
        
        // Build widget attributes with data-lang and data-base
        $widget_atts = array(
            'data-lang' => $lang,
            'data-base' => $currency
        );
        if (!empty($custom_class)) {
            $widget_atts['class'] = sanitize_html_class($custom_class);
        }

        $widget_attributes = '';
        foreach ($widget_atts as $key => $value) {
            $widget_attributes .= ' ' . $key . '="' . esc_attr($value) . '"';
        }

        // Enqueue script with parameters in footer
        $script_url = 'https://widget.pashizi.com/calculator?currency=' . urlencode($currency) . '&lang=' . urlencode($lang);
        wp_enqueue_script(
            'pashizi-widget-' . uniqid(),
            $script_url,
            array(),
            defined('PASHIZI_WIDGET_VERSION') ? PASHIZI_WIDGET_VERSION : '1.0.0',
            true // Load in footer (end of body)
        );

    echo '<div data-pashizi-widget' . wp_kses_post($widget_attributes) . '></div>';
        
    echo wp_kses_post($args['after_widget']);
    }
    
    /**
     * Widget form
     */
    public function form($instance) {
        $title = !empty($instance['title']) ? $instance['title'] : '';
        $currency = !empty($instance['currency']) ? $instance['currency'] : 'usd';
        $lang = !empty($instance['lang']) ? $instance['lang'] : 'en';
        $custom_class = !empty($instance['custom_class']) ? $instance['custom_class'] : '';
        
        $currencies = $this->get_supported_currencies();
        $languages = $this->get_supported_languages();
        ?>
        <p>
            <label for="<?php echo esc_attr($this->get_field_id('title')); ?>"><?php esc_html_e('Title:', 'pashizi-widget'); ?></label>
            <input class="widefat" id="<?php echo esc_attr($this->get_field_id('title')); ?>" name="<?php echo esc_attr($this->get_field_name('title')); ?>" type="text" value="<?php echo esc_attr($title); ?>">
        </p>
        
        <p>
            <label for="<?php echo esc_attr($this->get_field_id('currency')); ?>"><?php esc_html_e('Currency:', 'pashizi-widget'); ?></label>
            <select class="widefat" id="<?php echo esc_attr($this->get_field_id('currency')); ?>" name="<?php echo esc_attr($this->get_field_name('currency')); ?>">
                <?php foreach ($currencies as $code => $name): ?>
                    <option value="<?php echo esc_attr($code); ?>" <?php selected($currency, $code); ?>><?php echo esc_html($name); ?></option>
                <?php endforeach; ?>
            </select>
        </p>
        
        <p>
            <label for="<?php echo esc_attr($this->get_field_id('lang')); ?>"><?php esc_html_e('Language:', 'pashizi-widget'); ?></label>
            <select class="widefat" id="<?php echo esc_attr($this->get_field_id('lang')); ?>" name="<?php echo esc_attr($this->get_field_name('lang')); ?>">
                <?php foreach ($languages as $code => $name): ?>
                    <option value="<?php echo esc_attr($code); ?>" <?php selected($lang, $code); ?>><?php echo esc_html($name); ?></option>
                <?php endforeach; ?>
            </select>
        </p>
        
        <p>
            <label for="<?php echo esc_attr($this->get_field_id('custom_class')); ?>"><?php esc_html_e('Custom CSS Class:', 'pashizi-widget'); ?></label>
            <input class="widefat" id="<?php echo esc_attr($this->get_field_id('custom_class')); ?>" name="<?php echo esc_attr($this->get_field_name('custom_class')); ?>" type="text" value="<?php echo esc_attr($custom_class); ?>">
            <small><?php esc_html_e('Optional: Add custom CSS class for styling.', 'pashizi-widget'); ?></small>
        </p>
        <?php
    }
    
    /**
     * Update widget settings
     */
    public function update($new_instance, $old_instance) {
        $instance = array();
        $instance['title'] = (!empty($new_instance['title'])) ? sanitize_text_field($new_instance['title']) : '';
        $instance['currency'] = (!empty($new_instance['currency'])) ? sanitize_text_field($new_instance['currency']) : 'usd';
        $instance['lang'] = (!empty($new_instance['lang'])) ? sanitize_text_field($new_instance['lang']) : 'en';
        $instance['custom_class'] = (!empty($new_instance['custom_class'])) ? sanitize_html_class($new_instance['custom_class']) : '';
        
        return $instance;
    }
    
    /**
     * Get supported currencies
     */
    private function get_supported_currencies() {
        return array(
            'usd' => 'USD - US Dollar',
            'eur' => 'EUR - Euro',
            'gbp' => 'GBP - British Pound',
            'jpy' => 'JPY - Japanese Yen',
            'chf' => 'CHF - Swiss Franc',
            'cad' => 'CAD - Canadian Dollar',
            'aud' => 'AUD - Australian Dollar',
            'sgd' => 'SGD - Singapore Dollar',
            'hkd' => 'HKD - Hong Kong Dollar',
            'nok' => 'NOK - Norwegian Krone',
            'sek' => 'SEK - Swedish Krona',
            'dkk' => 'DKK - Danish Krone',
            'myr' => 'MYR - Malaysian Ringgit',
            'rub' => 'RUB - Russian Ruble',
            'try' => 'TRY - Turkish Lira',
            'inr' => 'INR - Indian Rupee',
            'cny' => 'CNY - Chinese Yuan',
            'aed' => 'AED - UAE Dirham',
            'thb' => 'THB - Thai Baht',
            'qar' => 'QAR - Qatari Riyal',
            'sar' => 'SAR - Saudi Riyal',
            'omr' => 'OMR - Omani Rial',
            'bhd' => 'BHD - Bahraini Dinar',
            'kwd' => 'KWD - Kuwaiti Dinar',
            'iqd' => 'IQD - Iraqi Dinar',
            'azn' => 'AZN - Azerbaijani Manat',
            'afn' => 'AFN - Afghan Afghani',
            'amd' => 'AMD - Armenian Dram'
        );
    }
    
    /**
     * Get supported languages
     */
    private function get_supported_languages() {
        return array(
            'en' => 'English',
            'fa' => 'Persian (فارسی)',
            'de' => 'German (Deutsch)',
            'ar' => 'Arabic (العربية)',
            'es' => 'Spanish (Español)',
            'fr' => 'French (Français)',
            'it' => 'Italian (Italiano)',
            'nl' => 'Dutch (Nederlands)',
            'ru' => 'Russian (Русский)',
            'zh' => 'Chinese (中文)'
        );
    }
}
