<?php
/**
 * Plugin Name: Pashizi Currency Exchange Widget
 * Plugin URI: https://widget.pashizi.com/calculator
 * Description: Display real-time currency exchange rates with an interactive calculator. Supports multiple currencies and languages with Persian (Farsi) localization.
 * Version: 1.0.0
 * Author: Pashizi
 * Author URI: https://www.pashizi.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: pashizi-widget
 * Domain Path: /languages
 * Requires at least: 5.0
 * Tested up to: 6.4
 * Requires PHP: 7.4
 * Network:
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Include required files
require_once plugin_dir_path(__FILE__) . 'includes/widget.php';

// Define plugin constants
define('PASHIZI_WIDGET_VERSION', '1.0.0');
define('PASHIZI_WIDGET_PLUGIN_URL', plugin_dir_url(__FILE__));
define('PASHIZI_WIDGET_PLUGIN_PATH', plugin_dir_path(__FILE__));
define('PASHIZI_WIDGET_PLUGIN_FILE', __FILE__);

/**
 * Main Pashizi Widget Plugin Class
 */
class PashiziWidget {
    
    /**
     * Instance of this class
     */
    private static $instance = null;
    
    /**
     * Get instance of this class
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        add_action('init', array($this, 'init'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        
        // Admin hooks
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'admin_init'));
        add_action('admin_enqueue_scripts', array($this, 'admin_enqueue_scripts'));
        
        // Shortcode
        add_shortcode('pashizi_widget', array($this, 'shortcode'));
        
        // Widget
        add_action('widgets_init', array($this, 'register_widget'));
        
        // Gutenberg block
        add_action('init', array($this, 'register_gutenberg_block'));
        
        // Activation/Deactivation hooks
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
    }
    
    /**
     * Initialize plugin
     */
    public function init() {
    // No need to manually load text domain since WP 4.6
    }
    
    /**
     * Enqueue scripts and styles
     */
    public function enqueue_scripts() {
        // Only load on pages that have the widget
        if ($this->should_load_scripts()) {
            wp_enqueue_script(
                'pashizi-widget-script',
                'https://widget.pashizi.com/calculator',
                array(),
                PASHIZI_WIDGET_VERSION,
                true // Load in footer (end of body)
            );
        }
    }
    
    /**
     * Enqueue admin scripts and styles
     */
    public function admin_enqueue_scripts($hook) {
        // Only load on plugin settings page
        if ($hook == 'settings_page_pashizi-widget') {
            wp_enqueue_style(
                'pashizi-widget-admin-css',
                PASHIZI_WIDGET_PLUGIN_URL . 'assets/css/admin.css',
                array(),
                PASHIZI_WIDGET_VERSION
            );
        }
    }
    
    /**
     * Check if scripts should be loaded
     */
    private function should_load_scripts() {
        global $post;
        
        // Check if shortcode is present in content
        if ($post && has_shortcode($post->post_content, 'pashizi_widget')) {
            return true;
        }
        
        // Check if widget is active
        if (is_active_widget(false, false, 'pashizi_widget')) {
            return true;
        }
        
        // Check if Gutenberg block is present
        if ($post && has_block('pashizi/currency-widget', $post)) {
            return true;
        }
        
        return false;
    }
    
    /**
     * Shortcode handler
     */
    public function shortcode($atts) {
        $atts = shortcode_atts(array(
            'currency' => '',
            'lang' => '',
            'class' => '',
            'id' => ''
        ), $atts, 'pashizi_widget');

        // Fallback to defaults if not specified
        $currency = !empty($atts['currency']) ? sanitize_text_field($atts['currency']) : 'usd';
        $lang = !empty($atts['lang']) ? sanitize_text_field($atts['lang']) : 'en';
        $class = sanitize_html_class($atts['class']);
        $id = sanitize_html_class($atts['id']);
        
        // Build widget HTML with data-lang and data-base attributes
        $widget_atts = array(
            'data-lang' => $lang,
            'data-base' => $currency
        );
        if (!empty($class)) {
            $widget_atts['class'] = $class;
        }
        if (!empty($id)) {
            $widget_atts['id'] = $id;
        }

        $widget_attributes = '';
        foreach ($widget_atts as $key => $value) {
            $widget_attributes .= ' ' . $key . '="' . esc_attr($value) . '"';
        }

        // Enqueue script with parameters in footer
        $script_url = 'https://widget.pashizi.com/calculator?currency=' . urlencode($currency) . '&lang=' . urlencode($lang);
        wp_enqueue_script(
            'pashizi-widget-' . uniqid(),
            $script_url,
            array(),
            PASHIZI_WIDGET_VERSION,
            true // Load in footer
        );

        return '<div data-pashizi-widget' . $widget_attributes . '></div>';
    }
    
    /**
     * Register WordPress widget
     */
    public function register_widget() {
        register_widget('Pashizi_Currency_Widget');
    }
    
    /**
     * Register Gutenberg block
     */
    public function register_gutenberg_block() {
        if (function_exists('register_block_type')) {
            wp_register_script(
                'pashizi-widget-block-script',
                PASHIZI_WIDGET_PLUGIN_URL . 'assets/js/block.js',
                array('wp-blocks', 'wp-element', 'wp-components', 'wp-editor'),
                PASHIZI_WIDGET_VERSION,
                true
            );
            
            register_block_type('pashizi/currency-widget', array(
                'editor_script' => 'pashizi-widget-block-script',
                'render_callback' => array($this, 'render_block'),
                'attributes' => array(
                    'currency' => array(
                        'type' => 'string',
                        'default' => 'usd'
                    ),
                    'lang' => array(
                        'type' => 'string',
                        'default' => 'en'
                    )
                )
            ));
        }
    }
    
    /**
     * Render Gutenberg block
     */
    public function render_block($attributes) {
        $currency = isset($attributes['currency']) ? $attributes['currency'] : 'usd';
        $lang = isset($attributes['lang']) ? $attributes['lang'] : 'en';
        
        return $this->shortcode(array(
            'currency' => $currency,
            'lang' => $lang
        ));
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_options_page(
            esc_html__('Pashizi Widget Settings', 'pashizi-widget'),
            esc_html__('Pashizi Widget', 'pashizi-widget'),
            'manage_options',
            'pashizi-widget',
            array($this, 'admin_page')
        );
    }
    
    /**
     * Initialize admin settings
     */
    public function admin_init() {
    register_setting('pashizi_widget_settings', 'pashizi_widget_options', array('sanitize_callback' => 'sanitize_text_field'));
        
        add_settings_section(
            'pashizi_widget_main',
            esc_html__('Widget Settings', 'pashizi-widget'),
            array($this, 'settings_section_callback'),
            'pashizi-widget'
        );
        
        add_settings_field(
            'default_currency',
            esc_html__('Default Currency', 'pashizi-widget'),
            array($this, 'currency_field_callback'),
            'pashizi-widget',
            'pashizi_widget_main'
        );
        
        add_settings_field(
            'default_language',
            esc_html__('Default Language', 'pashizi-widget'),
            array($this, 'language_field_callback'),
            'pashizi-widget',
            'pashizi_widget_main'
        );
    }
    
    /**
     * Settings section callback
     */
    public function settings_section_callback() {
    echo '<p>' . esc_html__('Configure default settings for the Pashizi currency widget.', 'pashizi-widget') . '</p>';
    }
    
    /**
     * Currency field callback
     */
    public function currency_field_callback() {
        $options = get_option('pashizi_widget_options');
        $value = isset($options['default_currency']) ? $options['default_currency'] : 'usd';
        
        $currencies = $this->get_supported_currencies();
        
        echo '<select name="pashizi_widget_options[default_currency]">';
        foreach ($currencies as $code => $name) {
            echo '<option value="' . esc_attr($code) . '"' . selected($value, $code, false) . '>' . esc_html($name) . '</option>';
        }
        echo '</select>';
    echo '<p class="description">' . esc_html__('Select the default currency for new widgets.', 'pashizi-widget') . '</p>';
    }
    
    /**
     * Language field callback
     */
    public function language_field_callback() {
        $options = get_option('pashizi_widget_options');
        $value = isset($options['default_language']) ? $options['default_language'] : 'en';
        
        $languages = $this->get_supported_languages();
        
        echo '<select name="pashizi_widget_options[default_language]">';
        foreach ($languages as $code => $name) {
            echo '<option value="' . esc_attr($code) . '"' . selected($value, $code, false) . '>' . esc_html($name) . '</option>';
        }
        echo '</select>';
    echo '<p class="description">' . esc_html__('Select the default language for new widgets.', 'pashizi-widget') . '</p>';
    }
    
    /**
     * Admin page
     */
    public function admin_page() {
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            
            <div class="pashizi-widget-admin">
                <div class="pashizi-widget-admin-main">
                    <form method="post" action="options.php">
                        <?php
                        settings_fields('pashizi_widget_settings');
                        do_settings_sections('pashizi-widget');
                        submit_button();
                        ?>
                    </form>
                </div>
                
                <div class="pashizi-widget-admin-sidebar">
                    <div class="pashizi-widget-info-box">
                        <h3><?php esc_html_e('Usage', 'pashizi-widget'); ?></h3>
                        <h4><?php esc_html_e('Shortcode', 'pashizi-widget'); ?></h4>
                        <p><?php esc_html_e('Use the shortcode in posts and pages:', 'pashizi-widget'); ?></p>
                        <code>[pashizi_widget currency="usd" lang="en"]</code>
                        
                        <h4><?php esc_html_e('Widget', 'pashizi-widget'); ?></h4>
                        <p><?php esc_html_e('Go to Appearance > Widgets and add the Pashizi Currency Widget.', 'pashizi-widget'); ?></p>
                        
                        <h4><?php esc_html_e('Gutenberg Block', 'pashizi-widget'); ?></h4>
                        <p><?php esc_html_e('Search for "Pashizi Currency" in the block editor.', 'pashizi-widget'); ?></p>
                    </div>
                    
                    <div class="pashizi-widget-info-box">
                        <h3><?php esc_html_e('Supported Currencies', 'pashizi-widget'); ?></h3>
                        <p><?php esc_html_e('USD, EUR, GBP, JPY, CHF, CAD, AUD, SGD, HKD, NOK, SEK, DKK, MYR, RUB, TRY, INR, CNY, AED, THB, QAR, SAR, OMR, BHD, KWD, IQD, AZN, AFN, AMD', 'pashizi-widget'); ?></p>
                    </div>
                    
                    <div class="pashizi-widget-info-box">
                        <h3><?php esc_html_e('Supported Languages', 'pashizi-widget'); ?></h3>
                        <p><?php esc_html_e('English, Persian (Farsi), German, Arabic, Spanish, French, Italian, Dutch, Russian, Chinese', 'pashizi-widget'); ?></p>
                    </div>
                </div>
            </div>
        </div>
        
        <style>
        .pashizi-widget-admin {
            display: flex;
            gap: 20px;
            margin-top: 20px;
        }
        .pashizi-widget-admin-main {
            flex: 2;
        }
        .pashizi-widget-admin-sidebar {
            flex: 1;
        }
        .pashizi-widget-info-box {
            background: #fff;
            border: 1px solid #ccd0d4;
            box-shadow: 0 1px 1px rgba(0,0,0,.04);
            padding: 15px;
            margin-bottom: 20px;
        }
        .pashizi-widget-info-box h3 {
            margin-top: 0;
        }
        .pashizi-widget-info-box code {
            display: block;
            background: #f6f7f7;
            padding: 10px;
            margin: 10px 0;
            border-radius: 3px;
        }
        </style>
        <?php
    }
    
    /**
     * Get supported currencies
     */
    private function get_supported_currencies() {
        return array(
            'usd' => 'USD - US Dollar',
            'eur' => 'EUR - Euro',
            'gbp' => 'GBP - British Pound',
            'jpy' => 'JPY - Japanese Yen',
            'chf' => 'CHF - Swiss Franc',
            'cad' => 'CAD - Canadian Dollar',
            'aud' => 'AUD - Australian Dollar',
            'sgd' => 'SGD - Singapore Dollar',
            'hkd' => 'HKD - Hong Kong Dollar',
            'nok' => 'NOK - Norwegian Krone',
            'sek' => 'SEK - Swedish Krona',
            'dkk' => 'DKK - Danish Krone',
            'myr' => 'MYR - Malaysian Ringgit',
            'rub' => 'RUB - Russian Ruble',
            'try' => 'TRY - Turkish Lira',
            'inr' => 'INR - Indian Rupee',
            'cny' => 'CNY - Chinese Yuan',
            'aed' => 'AED - UAE Dirham',
            'thb' => 'THB - Thai Baht',
            'qar' => 'QAR - Qatari Riyal',
            'sar' => 'SAR - Saudi Riyal',
            'omr' => 'OMR - Omani Rial',
            'bhd' => 'BHD - Bahraini Dinar',
            'kwd' => 'KWD - Kuwaiti Dinar',
            'iqd' => 'IQD - Iraqi Dinar',
            'azn' => 'AZN - Azerbaijani Manat',
            'afn' => 'AFN - Afghan Afghani',
            'amd' => 'AMD - Armenian Dram'
        );
    }
    
    /**
     * Get supported languages
     */
    private function get_supported_languages() {
        return array(
            'en' => 'English',
            'fa' => 'Persian (فارسی)',
            'de' => 'German (Deutsch)',
            'ar' => 'Arabic (العربية)',
            'es' => 'Spanish (Español)',
            'fr' => 'French (Français)',
            'it' => 'Italian (Italiano)',
            'nl' => 'Dutch (Nederlands)',
            'ru' => 'Russian (Русский)',
            'zh' => 'Chinese (中文)'
        );
    }
    
    /**
     * Plugin activation
     */
    public function activate() {
        // Set default options
        $default_options = array(
            'default_currency' => 'usd',
            'default_language' => 'en'
        );
        
        if (!get_option('pashizi_widget_options')) {
            add_option('pashizi_widget_options', $default_options);
        }
    }
    
    /**
     * Plugin deactivation
     */
    public function deactivate() {
        // Clean up if needed
    }
}

// Initialize plugin
PashiziWidget::get_instance();
